//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.4

import wesual.data.Core 1.0

ListView {
    id : listView

    property alias scrollAnimation : scrollAnimation

    // -------------------------------------------------------------------------
    //! Selection properties
    property var    selection      :  []
    //! previously selected index, used as anchor for multiselection
    property int    previousIndex  : -1
    property bool   multiSelection : true

    // -------------------------------------------------------------------------
    //! Key Handling Signals
    signal deletePressed()
    signal escapePressed()
    signal leftPressed()
    signal rightPressed()
    signal upPressed()
    signal downPressed()

    delegate : SelectionListDelegate {}

    //! range selection function
    function selectRange(from, to, reselect) {
        if (from === -1 || to === -1)
            return [];

        if (from > to) {
            var tmp = from;
            from = to;
            to = tmp;
        }

        var docs = listView.model.documents;
        if (docs.length === 0)
            return [];

        if (to >= docs.length)
            to = docs.length-1;

        reselect = typeof reselect !== 'undefined' ? reselect : true;

        var newSelection = [];
        for (var i = from; i <= to; ++i) {
            if (reselect || listView.selection.indexOf(docs[i]) === -1)
                newSelection.push(docs[i]);
        }
        return newSelection;
    }

    //! Key Handling
    function initSelection(event) {
        listView.currentIndex = listView.model.documents.length > 0 ? 0 : -1;
        listView.previousIndex = -1;
        listView.selection = [model.documents[currentIndex]];
        event.accepted = true;
    }

    function deselect() {
        listView.currentIndex = -1;
        listView.previousIndex = -1;
        listView.selection = [];
    }

    boundsBehavior : Flickable.StopAtBounds

    move : Transition {
        SmoothedAnimation {
            properties : "x,y"
            duration: 200
        }
    }
    displaced : move
    remove : Transition {
        NumberAnimation {
            duration : 150
            properties : "opacity,scale"
            from : 1
            to : 0.01
            alwaysRunToEnd : true
        }
    }
    add : Transition {
        SequentialAnimation {
            PropertyAction {
                property : "opacity"
                value : 0
            }
            PauseAnimation { duration : 100 }
            NumberAnimation {
                duration : 150
                property : "opacity"
                from : 0
                to : 1
                alwaysRunToEnd : true
            }
        }
    }
    populate : add

    NumberAnimation {
        id : scrollAnimation
        target : listView
        property : listView.orientation === Qt.Horizontal ?
                       "contentX" : "contentY"
        easing.type : Easing.Linear
    }

    focus : true
    keyNavigationWraps : true
    Keys.onPressed : {
        var docs = listView.model.documents;
        var shiftMultiSelection = (event.modifiers & Qt.ShiftModifier)
                && listView.multiSelection;

         if (event.matches(StandardKey.SelectAll) && listView.multiSelection) {
            listView.previousIndex = docs.length > 0 ? 0 : -1;
            listView.currentIndex = docs.length-1;
            listView.selection = selectRange(0, (docs.length-1));
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Up && shiftMultiSelection) {
            if (listView.selection.length === 0) {
                upPressed();
                initSelection(event);
                return;
            }

            if (listView.previousIndex === -1)
                listView.previousIndex = listView.currentIndex;

            listView.decrementCurrentIndex();
            listView.selection = listView.selectRange(listView.previousIndex,
                                                      listView.currentIndex);
            upPressed();
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Down && shiftMultiSelection){

            if(listView.selection.length === 0) {
                downPressed();
                initSelection(event);
                return;
            }

            if (listView.previousIndex === -1)
                listView.previousIndex = listView.currentIndex;

            listView.incrementCurrentIndex();
            listView.selection = listView.selectRange(listView.previousIndex,
                                                      listView.currentIndex);
            downPressed();
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Escape) {
            escapePressed();
            listView.selection = [];
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Delete) {
            deletePressed();
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Right){
            rightPressed();
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Left){
            leftPressed();
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Up) {
            if (listView.selection.length === 0) {
                upPressed();
                initSelection(event);
                return;
            }

            listView.decrementCurrentIndex();
            listView.selection = [docs[currentIndex]];
            upPressed();
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Down) {
            if (listView.selection.length === 0) {
                downPressed();
                initSelection(event);
                return;
            }

            listView.incrementCurrentIndex();
            listView.selection = [docs[currentIndex]];
            downPressed();
            event.accepted = true;
            return;
        }

    } //key pressed
}

